//*************************************************************************************************
//
//	Description:
//		bodywork.fx - Vehicle bodywork shader.
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     06/19/2006  0.1           Created
//		TNettleship			07/10/2007	0.2						Added support for vertex alpha.
//		TNettleship			07/11/2007	0.3						Changed lighting to work in world coords.
//		TNettleship     07/24/2007  0.4           Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  0.5						Removed texture bias.
//		TNettleship     08/17/2007  0.6						Added more specialisation.
//		TNettleship     10/03/2007  0.7						Fixed problems with envmap sparkling.
//		TNettleship			10/23/2007	0.8						Converted to on-load rendermode behaviour binding.
//		TNettleship			01/23/2008 	0.9						Improvements to 'sparkling' problems.
//		MGriffiths			03/04/2009 	1.0						Added flip and candy effects
//		MGriffiths			10/04/2009 	1.1						Added metalflake effect
//	<TABLE>
//
//*************************************************************************************************


#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
//#define BONNETCAM_ENVMAP
#define USE_FRESNEL
//#define METALLIC
//#define USE_LAYER_2
#define NORMAL_MAPPING
#define DIRT_SCRATCH
#define USE_VINYLS
#define ALLOW_VINYLS
//#define FLIPCANDY
//#define METALFLAKE
//#define PEARLESCENT
#endif

//#define PEARLESCENT
//#define METALFLAKE
//#define METALLIC
//#define MATTE

// We use preprocessor definitions instead of variables in this shader, to ensure consistent bodywork values across all cars

#if defined( MATTE ) 
#define GLOBAL_ENVMAP_FACTOR 0.45f
#define GLOBAL_SPECULAR_FACTOR 0.115f
#define GLOBAL_SPECULAR_POWER 20.0f
#define SUNSPOT_SPECULAR_POWER 0.0f
#define SUNSPOT_BRIGHTNESS_MULT 0.0f
#else
#define GLOBAL_ENVMAP_FACTOR 1.0f
#define GLOBAL_SPECULAR_FACTOR 0.035f
#define GLOBAL_SPECULAR_POWER 20.0f
#define SUNSPOT_SPECULAR_POWER 8192.0f
#define SUNSPOT_BRIGHTNESS_MULT 4.0f
#endif


// Uncomment to use experimental higher quality envmapping approach
//#define EXPERIMENTAL_ENVMAPPING


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
float4x4 view : View
<
	bool appEdit = false;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// Primary UV mapping channel, used for the decal layer colour
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Use to define the clamping coordinate range for the decal layer mapping
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Used for the base colour
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool export = false;
> = 0;

// Used for the dirt and noise mapping - must be no overlaps
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 4;
	int RuntimeTexcoord = 3;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord5 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 5;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Surface behaviour
//

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif


SPECIALISATION_PARAM( metallic, "Metallic?", "METALLIC" )	// TRUE if metallic paintwork calcs should be used

#if defined( _3DSMAX_ ) || defined( METALLIC )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( metallicColour, "Accent Colour", metallic, 1.0f, 1.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_DEFAULTED( metallicPower, "Metallic Power", metallic, 8.0f )
#endif


SPECIALISATION_PARAM( metalflake, "MetalFlake?", "METALFLAKE" )	// TRUE if metalflake calcs should be used

#if defined( _3DSMAX_ ) || defined( METALFLAKE )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( fleckMaterialColour, "Flecks Colour", metalflake, 0.5f, 0.5f, 0.5f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fleckScale, "Flecks Scale", metalflake, 0.0f, 100000.0f, 2.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fleckLocality, "Flecks locality", metalflake, 0.0f, 20.0f, 7.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fleckShininess, "Flecks shininess", metalflake, 0.0f, 50.0f, 4.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fleckNoiseFrequency, "Flecks frequency", metalflake, 0.0f, 10.0f, 2.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fleckClearcoatSmoothness, "Flecks clear coat smoothness", metalflake, 0.0f, 100.0f, 6.5f )

#endif


SPECIALISATION_PARAM( pearlescent, "Pearlescent?", "PEARLESCENT" )	// TRUE if pearlescent calcs should be used

#if defined( _3DSMAX_ ) || defined( PEARLESCENT )

DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( pearlColour, "Pearl Colour", pearlescent, 0.0625f, 0.185f, 0.177f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( pearlSpecularScaleR, "Pearl Specular Scale Red", pearlescent, 0.0f, 100000.0f, 12.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( pearlSpecularScaleG, "Pearl Specular Scale Green", pearlescent, 0.0f, 100000.0f, 11.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( pearlSpecularScaleB, "Pearl Specular Scale Blue", pearlescent, 0.0f, 100000.0f, 7.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( pearlRollOff, "Pearl RollOff", pearlescent, 0.0f, 1.0f, 0.05f )

#endif


SPECIALISATION_PARAM( flipcandy, "FlipCandy?", "FLIPCANDY" )	// TRUE if flip/candy paintwork calcs should be used

#if defined( _3DSMAX_ ) || defined( FLIPCANDY )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( flipColour, "Flip Accent Colour", flipcandy, 1.0f, 1.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_DEFAULTED( flipPower, "Flip Power", flipcandy, 1.5f )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( candyColour, "Candy Accent Colour", flipcandy, 1.0f, 1.0f, 1.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_DEFAULTED( candyPower, "Candy Power", flipcandy, 1.5f )
#endif


SPECIALISATION_PARAM( matte, "Matte?", "MATTE" )	// TRUE if matte paintwork calcs should be used

#if defined( _3DSMAX_ ) || defined( MATTE )
DEPENDENT_FLOAT_PARAM_DEFAULTED( specularTextureAlpha, "specular Texture Alpha", matte, 0.0f )
#endif


// Vinyls (custom livery).
//
// This option mostly matches the same option in glass.fx,
// although the base for the "painted livery" support is inherent to bodywork.fx.
// The option for vinyls enhances the behaviour of conventional bodywork,
// it adds special feature of using alpha channel of diffuse texture to choose between bodywork and vinyls.
// Note that USE_VINYLS is incompatible with premultiplied alpha (see below).

SPECIALISATION_PARAM( useVinyls, "Use vinyls?", "USE_VINYLS" )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsSpecularFactor, "Specular Factor of Vinyls", useVinyls, 0.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsSpecularPower, "Specular Power of Vinyls (min/max)", useVinyls, 0.0f, 65536.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( vinylsEMapFactor, "EMap Factor of Vinyls", useVinyls, 0.0f, 16.0f, 0.5f )
#endif

SPECIALISATION_PARAM( allowVinyls, "Allow vinyls?", "ALLOW_VINYLS" )


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha (see also USE_VINYLS).
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha (see also USE_VINYLS).
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useLayer2, "Use layer 2?", "USE_LAYER_2" )	// TRUE if layer 2 should be used
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord0Dependency, texCoord0, TEXCOORD0, useLayer2 )
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord1Dependency, texCoord1, TEXCOORD2, useLayer2 )

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
DEPENDENT_TEXTURE_PARAM( diffuseTextureL2, "Diff Texture L2", useLayer2 )
DEPENDENT_TEXTURE_PARAM( specularTextureL2, "Spec Texture L2", useLayer2 )
DEPENDENT_TEXTURE_PARAM( clampControlTexture, "Clamp Control Tex", useLayer2 )
#endif


SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if normal mapping should be used
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( NORMAL_MAPPING ) )
BOTH_DEPENDENT_TEXTURE_PARAM( normalTextureL2, "Normal Tex L2", useLayer2;useNormalMap )
#endif

#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_FRESNEL ) )
BOTH_DEPENDENT_TEXTURE_PARAM( fresnelTextureL2, "Fresnel Control L2", useLayer2;useFresnel )
#endif

#if defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )
texture environmentTexture : TEXTURE					// Environment map (RGB)
<
	string UIName = "Env Texture";
	string Type = "CUBE";
	bool appEdit = false;
	bool export = false;
>;

#else

texture motionBlurTexture : motionBlurTexture					// Bonnetcam environment map uses the motion blur texture
<
	bool appEdit = false;
>;

#endif

#if defined( _3DSMAX_ ) || defined( METALLIC )
EITHER_DEPENDENT_TEXTURE_PARAM( noiseTexture, "Noise Texture", metallic )
#endif


// 3D Noise Map

#if defined( _3DSMAX_ ) || defined(METALFLAKE)
texture noise3DTexture : TEXTURE
<
	bool appEdit = false;
	string Type = "VOLUME";
>;
//Texture3D noiseMap < string file="Render\Textures\noise3d.dds";>;
#endif


float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;

float globalEMapFactor
<
	string UIName = "EMap Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

#if defined( _3DSMAX_ )
#define GetLightDirection( _input ) _input.lightVector
#define GetLightColour( _input ) lightColour
#elif 		defined( _LIGHTING_D_ )		|| 	defined( _LIGHTING_DD_ )	|| 	defined( _LIGHTING_DDD_ ) || defined( _LIGHTING_DP_ )	||	defined( _LIGHTING_DS_ )	|| 	defined( _LIGHTING_DPS_ ) || defined( _LIGHTING_DPP_ )	|| 	defined( _LIGHTING_DSS_ ) || 	defined( _LIGHTING_DDP_ ) || defined( _LIGHTING_DDS_ ) 
#define GetLightDirection( _input ) lightDirection0
#define GetLightColour( _input ) lightColour0
#else
#define GetLightDirection( _input ) float3(0.0f,1.0f,0.0f)
#define GetLightColour( _input ) float4(1.0f, 1.0f, 1.0f, 1.0f)
#endif

//
// Dirt n scratches
//

SPECIALISATION_PARAM( dirtNscratch, "Dirt 'n' Scratch?", "DIRT_SCRATCH" )	// TRUE if dirt and scratches should be applied

DEPENDENT_TEXTURE_PARAM( scratchControlTexture, "Dirt / Scratch Tex.", dirtNscratch )			// multi-channel dirt and scratch texture, will be supplied by game
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( primerBasis, "Primer Colour", dirtNscratch, 0.7f, 0.7f, 0.7f, 1.0f  )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( metalBasis,  "Metal Colour",  dirtNscratch, 0.3f, 0.3f, 0.3f, 1.0f  )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( dirtBasis,  "Dirt Colour",  dirtNscratch, 0.233f, 0.203f, 0.100f, 1.0f  )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( noiseScale, "Dirt Noise Scale", dirtNscratch, 0.0f, 128.0f, 4.0f )
DEPENDENT_COLOURPICKER_PARAM_DEFAULTED( dirtScale,  "Set me blue",  dirtNscratch, 0.0f, 0.0f, 1.0f, 1.0f  )
DECLARE_DEPENDENT_VERTEX_STREAM( texcoord4Dependency, texCoord3, TEXCOORD4, dirtNscratch )

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


SPECIALISATION_PARAM( bonnetCamEnvMap, "Bonnetcam Envmap?", "BONNETCAM_ENVMAP" )	// TRUE if special-case bonnetcam envmap should be used



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
#if defined( USE_VINYLS)
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
#else
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
#endif
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
sampler2D diffuseMapL2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTextureL2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
> 
= sampler_state
{
	Texture = < diffuseTextureL2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D specularMapL2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTextureL2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
> 
= sampler_state
{
	Texture = < specularTextureL2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};
#endif


#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( NORMAL_MAPPING ) )
sampler2D normalMapL2 : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTextureL2"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
	bool signtex=true;
> 
= sampler_state
{
	Texture = < normalTextureL2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};
#endif


#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_FRESNEL ) )
sampler2D fresnelMapL2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="fresnelTextureL2"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
> 
= sampler_state
{
	Texture = < fresnelTextureL2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
sampler2D clampControlMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="clampControlTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
> 
= sampler_state
{
	Texture = < clampControlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#else

sampler2D motionBlurMap : SAMPLER 
< 
#if defined(_PS3_)
	SET_SRGB_TEXTURE
#else
	SET_LINEAR_TEXTURE
#endif
	bool appEdit = false; 
	string SamplerTexture="motionBlurTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT1";
> 
= sampler_state
{
	Texture = < motionBlurTexture >;
#if defined(SET_FX_SAMPLER_STATES)
#if defined(_PS3_)
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#endif

#if defined( _3DSMAX_ ) || defined( METALLIC )
sampler2D noiseMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="noiseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
	bool signtex=true;
> 
= sampler_state
{
	Texture = < noiseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( METALFLAKE)
sampler3D noise3DMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="noise3DTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Mirror";
	string AddressV  = "Mirror";
	string AddressW  = "Mirror";
	int MipMapLODBias = 0;	
	bool signtex=true;
> 
= sampler_state
{
	Texture = < noise3DTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Mirror;
	AddressV  = Mirror;
	AddressW  = Mirror;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined ( DIRT_SCRATCH )
sampler2D scratchControlMap : sample
<
	SET_LINEAR_TEXTURE
	bool appEdit = false;
	string SamplerTexture="scratchControlTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
	string texformat="COMPRESSED_RGBA_S3TC_DXT5";
> 
= sampler_state
{
	Texture = < scratchControlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};
#endif


//-----------------------------------------------------------------------
//
// Functions
//

/*
uv coords input in 0-1 range (normal)
extends input gives clamp rectangle, defined within 0..1 range:
R = umin
G = vmin
B = umax
A = vmax
*/

float2 CalculateClampedCoordinates( float2 _inputCoords, float4 _clampExtents )
{
	float2 clampMin = _clampExtents.rg;
	float2 clampMax = _clampExtents.ba;
	float2 scaledCoords = clampMin + ( _inputCoords * ( clampMax - clampMin ) );

	float2 output = clamp( scaledCoords, clampMin, clampMax );

	return output;
}



//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float calculated_fresnel(float3 _eye, float3 _normal, float rindexRatio)
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f ) / pow ( 1.0f + rindexRatio, 2.0f );

	return saturate( R0 + ( 1.0f - R0 ) * pow(1.0f - saturate( dot( -_eye, _normal ) ), 5.0f ) );
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
#ifdef _3DSMAX_
	float3 position : POSITION;														// Object space position
	float3 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord (decal map)
	float2 texCoord1 : TEXCOORD2;													// UV channel 2 texture coord (control map)
	float2 texCoord2 : TEXCOORD3;													// UV channel 3 texture coord (base colour)
	float2 texCoord3 : TEXCOORD4;													// UV channel 4 texture coord (dirt n scratch map)
	float3 alpha		 : TEXCOORD5;													// Vertex alpha
	float3 normal    : NORMAL;														// Object space normal
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space binormal
#else

	float3 position : POSITION;														// Object space position
	float4 colour    : COLOR0;														// Vertex colour
#if defined( USE_LAYER_2 )
	// These two vertex streams aren't needed without layer 2
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord (decal map)
	float2 texCoord1 : TEXCOORD1;													// UV channel 2 texture coord (control map)
#endif
	float2 texCoord2 : TEXCOORD2;													// UV channel 3 texture coord (base colour)
#if defined( DIRT_SCRATCH )
  float2 texCoord3 : TEXCOORD3;
#endif
	float3 normal    : NORMAL;														// Object space normal
#if ( defined( NORMAL_MAPPING ) && defined( USE_LAYER_2 ) )
	// These two vertex streams aren't needed without normal mapping or layer 2
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space binormal
#endif

#endif
};


// Output structure (also input to the fragment shader)
struct VSOUTPUT
{
	float4 position							: POSITION;								// View-coords position
	float4 colour								: TEXCOORD0;									// Vertex colour
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
	// These two vertex streams aren't needed without layer 2
	float4 texCoord01						: TEXCOORD1;							// UV coords for all layers
#endif
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	float4 texCoord23						: TEXCOORD2;							// UV coords for base texture plus dirt
#else
	float2 texCoord23						: TEXCOORD2;							// UV coords for base texture
#endif
	float4 normal_plus_fresnel	: TEXCOORD3;							// Normal vector (world space), with fresnel term in w
	float4 eye									: TEXCOORD4;							// Eye vector (world space)

#if defined( _3DSMAX_ ) || ( defined( NORMAL_MAPPING ) && defined( USE_LAYER_2 ) )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent							: TEXCOORD5;							// Tangent vector (world space)
	float3 binormal							: TEXCOORD6;							// Binormal (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT BodyworkVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
#if defined( _3DSMAX_ )
	_output.texCoord23 = float4(_input.texCoord2.x, _input.texCoord2.y, _input.texCoord3.x, 1.0f - _input.texCoord3.y ) ;
#else
	_output.texCoord23.xy = _input.texCoord2;
	#if defined ( DIRT_SCRATCH )
		_output.texCoord23.zw = _input.texCoord3.xy;
	#endif
#endif

	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )

#ifdef _3DSMAX_
		// By default, Max flips the V coord of texture channels. This is because it
		// uses a BL coord system, instead of D3D's TL system. Flipping works fine for
		// wrapped textures, but for clamped textures it means the default 0...1 texture
		// space rectangle is off the object. Using 1-v instead of -v is correct in
		// both cases so we have this piece of code here, and the colorchannel declaration
		// above to do things correctly.
		_output.texCoord01 = float4( _input.texCoord0.x, 1.0f - _input.texCoord0.y, _input.texCoord1.x, 1.0f - _input.texCoord1.y );
#else
		_output.texCoord01 = float4( _input.texCoord0, _input.texCoord1 );
#endif

#endif
	DEPENDENT_CODE_END( useLayer2 )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	DEPENDENT_CODE_START_AND( useLayer2, useNormalMap )
#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( NORMAL_MAPPING ) )
		_output.tangent	 = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END_AND( useLayer2, useNormalMap )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_plus_fresnel = float4( normal, calculated_fresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_plus_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )
	
	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour								: TEXCOORD0;									// Vertex colour
	float4 texCoord01						: TEXCOORD1;							// UV coords for all layers
	float4 texCoord23						: TEXCOORD2;							// UV coords for base texture and dirt
	float4 normal_plus_fresnel	: TEXCOORD3;							// Normal vector (world space), with fresnel term in w
	float4 eye									: TEXCOORD4;							// Eye vector (world space)
	float3 tangent							: TEXCOORD5;							// Tangent vector (world space)
	float3 binormal							: TEXCOORD6;							// Binormal (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour								: TEXCOORD0_centroid;						// Vertex colour
#if defined( USE_LAYER_2 )
	// These two vertex streams aren't needed without layer 2
	float4 texCoord01						: TEXCOORD1_centroid;				// UV coords for all layers
#endif
#if defined( DIRT_SCRATCH )	
	float4 texCoord23						: TEXCOORD2_centroid;				// UV coords for base texture and dirt
#else
	float2 texCoord23						: TEXCOORD2_centroid;				// UV coords for base texture
#endif
	float4 normal_plus_fresnel	: TEXCOORD3_centroid;				// Normal vector (world space), with fresnel term in w
	float4 eye									: TEXCOORD4_centroid;							// Eye vector (world space)

#if ( defined( NORMAL_MAPPING ) && defined( USE_LAYER_2 ) )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent							: TEXCOORD5_centroid;							// Tangent vector (world space)
	float3 binormal							: TEXCOORD6_centroid;							// Binormal (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

#ifdef _POST_PARSE_
PSOUTPUT BodyworkFragmentShader( PSINPUT _input )
#else
PSOUTPUT BodyworkFragmentShader( PSINPUT _input, uniform bool _premultiplyAlpha )
#endif
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Calculate curvature of the current pixel
	float3 maxCurvatures = max( abs( ddx( _input.normal_plus_fresnel ).xyz ), abs( ddy( _input.normal_plus_fresnel ).xyz ) );
	float curvature = saturate( max( maxCurvatures.x, max( maxCurvatures.y, maxCurvatures.z ) ) );
	
	float globalSpecularFactorValue = GLOBAL_SPECULAR_FACTOR;
	float minSpecPowerValue = GLOBAL_SPECULAR_POWER;
	float maxSpecPowerValue = GLOBAL_SPECULAR_POWER;
	float globalEMapFactorValue = GLOBAL_ENVMAP_FACTOR * ( 1.0f - curvature );
	float sunspotSpecularFactor = SUNSPOT_BRIGHTNESS_MULT;

	//
	// Read textures
	//

	float4 diffuseTexColourL1 = tex2D( diffuseMap, _input.texCoord23.xy );
	float4 specularTexColourL1 = tex2D( specularMap, _input.texCoord23.xy );

	//
	// Block (USE_VINYLS):
	//   Prepare various vinyl-related factors that will get used in blocks below.
	//

	float emapFactor = globalEMapFactorValue;						// On pure bodywork, use conventional material property to control the strength of reflections.
	float specularFactor = globalSpecularFactorValue;		// On pure bodywork, specular strength is controlled by glass material properties.
	float4 vinylsEMapColour = specularTexColourL1;	// On pure bodywork, specular colour is controlled by layer 1 texture.
	float vinylsOpacity = 0.0f;
	float vinylsMetallicScaler = 1.0f;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )

		// Save unmodified copy of vinyl opacity for potential later use.
		vinylsOpacity = diffuseTexColourL1.a;

		// Dirt & scratches are not affected by vinyls.

		// Metallic accent is lowered, because vinyls cover the underlying metal.
		vinylsMetallicScaler = 1.0f - vinylsOpacity;

		// Change the intensity of specular and reflections / envmapping where covered by vinyls to the one provided by 'vinyl properties'.
		emapFactor = lerp( globalEMapFactorValue, vinylsEMapFactor, vinylsOpacity );
		specularFactor = lerp( globalSpecularFactorValue, vinylsSpecularFactor, vinylsOpacity );

		// Use white specular, after all, vinyls are plastic.
		vinylsEMapColour = lerp( specularTexColourL1, float4( 1.0f, 1.0f, 1.0f, vinylsSpecularPower ), vinylsOpacity );
		specularTexColourL1.a = lerp( specularTexColourL1.a, vinylsSpecularPower, vinylsOpacity );

#endif
	DEPENDENT_CODE_END( useVinyls )

	//
	// Block (DIRT_SCRATCH):
	//   Compute various scaling factors depending on the local state of dirt and scratches.
	//

#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	float4 controlColour;
	float paintPercent;
	float glossPercent;
	float metalPercent;
	float primerPercent;
	float specularPercent;
	float dirtPercent;
  float2 scratchControlUV;	// used to simplify code, to reduce risk of errors
#endif
	
	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )

	scratchControlUV.xy = _input.texCoord23.zw;

	controlColour = tex2D( scratchControlMap, scratchControlUV.xy );	// get control map colours 
	
	paintPercent = saturate( (0.34f - controlColour.g) * 50.0f );	// note these values are interdependent - don't change them without speaking to RobD
	glossPercent = saturate( 1.00f - (controlColour.g * 10.0f) );	// gloss value drops sooner, so that we appear to scrape through the surface before going through the paint
	metalPercent = saturate( (controlColour.g - 0.65f) * 50.0f ); 
	primerPercent = (1.0f - (paintPercent + metalPercent));

	specularPercent = glossPercent + metalPercent * metalBasis.w;	// using metal colour alpha channel to control shininess of metal layer
	
	dirtPercent = controlColour.b;	// no messing with this one, its just as it is, a splotch of dirt, with variable thickness

	specularPercent *= smoothstep( 1.0f, 0.0f, dirtPercent * 4.0f );	// scale down specular if there is any dirt on it
#endif
	DEPENDENT_CODE_END( dirtNscratch )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColourL1.a *= _input.colour.a;

	//
	// Block (USE_LAYER_2):
	//   Read the textures forming layer 2.
	//

	float2 clampedCoords;
	float4 diffuseTexColourL2;
	float4 specularTexColourL2;

	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		// Calculate the clamped version of texCoord0
		clampedCoords = CalculateClampedCoordinates( _input.texCoord01.xy, tex2D( clampControlMap, _input.texCoord01.zw ) );

		diffuseTexColourL2 = tex2D( diffuseMapL2, clampedCoords );
		specularTexColourL2 = tex2D( specularMapL2, clampedCoords );
#endif
	DEPENDENT_CODE_END( useLayer2 )

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal_plus_fresnel.xyz );
  float3 eye = normalize( _input.eye.xyz );

	//
	// Block (FLIPCANDY)
	// Flip and Candy effects
	//

#if defined( _3DSMAX_ ) || defined( METALLIC ) || defined( FLIPCANDY ) || defined( MATTE )
	float ndotv = saturate( dot( TSnormal, eye ) );
#endif

	DEPENDENT_CODE_START( flipcandy )
#if defined( _3DSMAX_ ) || defined( FLIPCANDY )
	//float3 flipColour = float3(0.0f,0.3f,0.0f);
	//float3 candyColour = float3(0.0f,0.0f,0.3f);
	//float flipPower = 1.1f;
	//float candyPower = 1.1f;
	float3 _L = GetLightDirection(_input);
	float3 _H = normalize( _L + eye);
	float ndoth = saturate( dot( TSnormal, _H ) );
	float c1 = pow( ndotv, flipPower );			
	float c2 = pow( 1.0f - ndotv, flipPower );
	float c3 = pow( ndoth, candyPower );
	diffuseTexColourL1.rgb = c1 * diffuseTexColourL1.rgb + c2 * flipColour.rgb + c3 * candyColour.rgb;	
#endif
  DEPENDENT_CODE_END( flipcandy )

	//
	// Block (METALLIC):
	//   Two-tone paint effect.
	//

	// If metallic paint effect is required
	DEPENDENT_CODE_START( metallic )
#if defined( _3DSMAX_ ) || defined( METALLIC )

		// Modify base colour with metallic accent colour
		diffuseTexColourL1.rgb += vinylsMetallicScaler * metallicColour.rgb * pow( ndotv, metallicPower );
#endif
	DEPENDENT_CODE_END( metallic )

	//
	// Block (NORMAL_MAPPING && USE_LAYER_2):
	//   Normal mapping (decal layer)
	//

	float3 normal;

	// If normal map support is required
	DEPENDENT_CODE_START_AND( useLayer2, useNormalMap )
#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( NORMAL_MAPPING ) )
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMapL2, clampedCoords );
		float3 normalFromMap = ( normalMapColour.agb * 2.0f ) - float3( 256.0f/255.0f, 256.0f/255.0f, 1.0f );	// bias to allow flat planes

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE_AND( useLayer2, useNormalMap )
#if defined( _3DSMAX_ ) || !( defined( USE_LAYER_2 ) && defined( NORMAL_MAPPING ) )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END_AND( useLayer2, useNormalMap )


	//
	// Blend decal layer and base layer colours
	//

	float blendFactor;
	float4 diffuseTexColour;
	float4 specularTexColour;

	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		blendFactor = diffuseTexColourL2.a;
		diffuseTexColour = lerp( diffuseTexColourL1, diffuseTexColourL2, blendFactor );
		specularTexColour = lerp( specularTexColourL1, specularTexColourL2, blendFactor );
		vinylsEMapColour = lerp( specularTexColourL1, specularTexColourL2, blendFactor);
#endif
	DEPENDENT_CODE_ELSE( useLayer2 )
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_2 )
		blendFactor = 0.0f;
		diffuseTexColour = diffuseTexColourL1;
		specularTexColour = specularTexColourL1;
#endif
	DEPENDENT_CODE_END( useLayer2 )

  DEPENDENT_CODE_START( matte )
#if defined( _3DSMAX_ ) || defined( MATTE ) 
  specularTexColour.a = specularTextureAlpha;
#endif
  DEPENDENT_CODE_END( matte )
	//
	// Block (PEARLESCENT)
	// Pearl paint pigment effect
	//

	DEPENDENT_CODE_START( pearlescent )
#if defined( _3DSMAX_ ) || defined( PEARLESCENT )

	//float pearlSpecularScaleR = 12.0;
	//float pearlSpecularScaleG = 11.0f;
	//float pearlSpecularScaleB = 7.0f;		
	//float3 pearlColour =  float3(0.0625f,0.185f,0.177f);
	//float pearlRollOff = 0.05f;

	float3 _H = CalculateH(-GetLightDirection(_input), eye );		 
	float h_dot_n = saturate(dot(_H,normal) + pearlRollOff);  
  diffuseTexColour.r = lerp( diffuseTexColour.r, pearlColour.r, pow(h_dot_n,pearlSpecularScaleR * specularTexColour.a) );
  diffuseTexColour.g = lerp( diffuseTexColour.g, pearlColour.g, pow(h_dot_n,pearlSpecularScaleG * specularTexColour.a) );
  diffuseTexColour.b = lerp( diffuseTexColour.b, pearlColour.b, pow(h_dot_n,pearlSpecularScaleB * specularTexColour.a) );
  
  /*float v_dot_n = dot(eye,normal);
		float vd = pearlThickness / v_dot_n ;
		float3 lamRGB = float3(6,5,4);
    float3 offsetRGB = (float3)0;
    float p = 60; 
    float vdr = p;
    float pi = 3.1415926535f;
    float3 fringeRgb = 0.5*(sin(2*pi*(vd*vdr)/lamRGB + pi/2.0 + offsetRGB) + 1);
    specularTexColour.rgb *= fringeRgb * pearlColour; */
  
#endif
	DEPENDENT_CODE_END( pearlescent )


	// Once blended, we need to adjust vinyls-related factors - layer 2 is on top of vinyls.

	DEPENDENT_CODE_START_AND( useLayer2, useVinyls )
#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_VINYLS ) )
		emapFactor = lerp( emapFactor, globalEMapFactorValue, blendFactor );
		specularFactor = lerp( specularFactor, globalSpecularFactorValue, blendFactor );
		vinylsEMapColour = lerp( vinylsEMapColour, specularTexColourL2, blendFactor );
#endif
	DEPENDENT_CODE_END_AND( useLayer2, useVinyls )

	//
	// Block (DIRT_SCRATCH):
	//    Apply dirt on top of the bodywork, and engrave scratches.
	//

	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	// apply dirt and scratches
	float4 interimColour = lerp( diffuseTexColour, primerBasis, primerPercent);
	interimColour = lerp( interimColour, metalBasis, metalPercent);
	diffuseTexColour = lerp( interimColour, dirtBasis, dirtPercent);
	emapFactor *= specularPercent;
	specularFactor *= specularPercent;
#endif
  DEPENDENT_CODE_END( dirtNscratch )

	// Write back the specularfactor value to the variable which will be used in the lighting calculations
	globalSpecularFactorValue = specularFactor;


	//
	// Block:
	//   Calculate the Fresnel factor.
	//

	float fresnelCoefficient;

	// Basic coefficient
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		fresnelCoefficient = saturate( _input.normal_plus_fresnel.w );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		fresnelCoefficient = 1.0f;
#endif
	DEPENDENT_CODE_END( useFresnel )

	// Blend with L2 fresnel map
	DEPENDENT_CODE_START_AND( useLayer2, useFresnel )
#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_FRESNEL ) )
		float fresnelModulationL2 = tex2D( fresnelMapL2, clampedCoords ).g;
		fresnelCoefficient = lerp( 1.0f, fresnelCoefficient, fresnelModulationL2 );
#endif
	DEPENDENT_CODE_END_AND( useLayer2, useFresnel )

	//
	// Block:
	//   Premultiplied alpha.
	//

	if ( _premultiplyAlpha )
	{
		// Premultiply the diffuse RGB by the diffuse alpha
		diffuseTexColour.rgb *= diffuseTexColourL1.a;
	}

	//
	// Block:
	//   Ambient lighting.
	//

#if defined( PSEUDO_GI ) && !defined( BONNETCAM_ENVMAP )
	// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
	float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
	float4 accumulatedColour = diffuseTexColour * envmapambient;
#else
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif

	//
	// If we're dirt mapping, perturb the normal back towards the tangent space one
	// according to the dirt amount.
	//
	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
		normal = normalize( lerp( normal, TSnormal, saturate( dirtPercent * 4.0f ) ) );
#endif
  DEPENDENT_CODE_END( dirtNscratch )

	//
	// Block (METALFLAKE):
	//   Metallic flecks
	//
	
	DEPENDENT_CODE_START( metalflake )
#if defined( _3DSMAX_ ) || defined( METALFLAKE )

  // Fleck-specific test params
  //float3 fleckMaterialColour  = {0.5,0.5,0.5};
  //float  fleckLocality  = 7.0f;
  //float  fleckShininess  = 3.0;
  //float fleckScale   = 2.0f;
  //float fleckNoiseFrequency  = 2;
  //float fleckClearcoatSmoothness = 7;
    
	// Calculate object space coords position (used to drive the random noise)
	float3 reflVect = reflect(_input.eye.xyz,normal);
	float3 objCoordsPos= mul( float4( worldPos, 1.0f ), worldI ).xyz;
  float3 _L = -GetLightDirection(_input);
  // Scale the obj_coords to make things more sparkly
  float3 noise_coords = float3(objCoordsPos * fleckScale);
  //float3 fleck_normal0 = (tex3D(noise3DMap, noise_coords).rgb*2.0f)-1.0f;
  //float3 fleck_normal1 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency).rgb*2.0f)-1.0f;
  //float3 fleck_normal2 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency * fleckNoiseFrequency).rgb*2.0f)-1.0f;
  //float3 fleck_normal3 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency * fleckNoiseFrequency * fleckNoiseFrequency).rgb*2.0f)-1.0f;
  //fleck_normal = normalize(fleck_normal+normal);
			
	// generate fleck normal from 3d noisemap
  float3 fleck_normal0 = (tex3D(noise3DMap, noise_coords).rgb);
  float3 fleck_normal1 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency).rgb);
  float3 fleck_normal2 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency * fleckNoiseFrequency).rgb);
  float3 fleck_normal3 = (tex3D(noise3DMap, noise_coords * fleckNoiseFrequency * fleckNoiseFrequency * fleckNoiseFrequency).rgb);    
  float3 fleck_normal = (normalize(fleck_normal2+fleck_normal1+fleck_normal3+fleck_normal0)*2.0f)-1.0f;
  //float3 fleck_worldnormal = normalize(mul( float4(fleck_normal,0.0f),world).xyz);
  //float3 noise_vector = .5 * fleck_normal0 + .25 * fleck_normal1 +  .125 * fleck_normal2 + .0625 * fleck_normal3;
 	//float3 L = normalize(GetLightDirection(_input) - worldPos);
  float3 H = CalculateH(-GetLightDirection(_input), eye );
	//float3 normaltowardsH = normalize( lerp(normal,H,pow(fleck_intensity*0.25f,5.0f)) );
	//float fleck_specular = 0.0f; //pow(saturate(dot(eye,fleck_normal3)),5.0f); //pow( max ( dot( normaltowardsH, H ), 0.0000001f ), 5.0f );
  //float fleck_specular = CalculateSpecularLighting( fleck_normal, _L, eye , 35 ) ;
  //float self_shadow = 1.0f;//( n_dot_l > 0) ? 1 : 0;
  
   float  fleck_intensity = 	pow(saturate(abs(fleck_normal0.z)), fleckShininess) + 
          									pow(saturate(abs(fleck_normal1.z)), fleckShininess) + 
        										pow(saturate(abs(fleck_normal2.z)), fleckShininess) + 
       											pow(saturate(abs(fleck_normal3.z)), fleckShininess);
   float n_dot_l = saturate(dot(fleck_normal, _L));
  //normal = lerp(normal,H,saturate(fleck_intensity) );

 	
 	// this is slightly made-up, but helps add sparkle
 	
 	//float r_dot_l =  saturate(dot(reflVect, _L));
 	float r_dot_l =  saturate(dot((fleck_normal*normal),H));
  float fleck_specular = pow(r_dot_l, fleckLocality);

	// peturb current normal, so that subsequence lighting and env-mapping
	// is slightly distorted by the fleck normal

  normal = normalize(fleck_normal + fleckClearcoatSmoothness * normal);
       
	DEPENDENT_CODE_START( dirtNscratch )
#if defined( _3DSMAX_ ) || defined( DIRT_SCRATCH )
	//apply dirt and scratches
	fleck_intensity *= paintPercent;	// no paint, no flecks
#endif

  DEPENDENT_CODE_END( dirtNscratch )
	
	accumulatedColour.rgb = (accumulatedColour.rgb + (fleck_specular+fleck_intensity*n_dot_l)*fleckMaterialColour); 

#endif
	DEPENDENT_CODE_END( metalflake )

	//
	// Block:
	//   Environment mapping or bonnet reflection (BONNETCAM_ENVMAP).
	//

	// Standard envmapping version
	float4 environmentTexColour;
#if defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )

	// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
	float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

#if defined( EXPERIMENTAL_ENVMAPPING )
	// Intersect the reflection vector with a 4m sphere centred on the car
	float3 envMapSpacePos = mul( float4( worldPos, 1.0f ), envTransformI ).xyz;
	float3 rayOrigin = envMapSpacePos;
	float3 rayDirection = reflectionVector;
	float B = dot( rayDirection, rayOrigin ) * 2.0f;
	float C = dot( rayOrigin, rayOrigin ) - ( 4.0f * 4.0f );
	float t0 = -B - sqrt( B*B - 4.0f * C ) / 2.0f;
	float t1 = -B + sqrt( B*B - 4.0f * C ) / 2.0f;
	float3 sphereIntersection = rayOrigin + ( rayDirection * max( t0, t1 ) );

	// Use the intersection point to look up the cubemap, not the reflection vector
	reflectionVector = normalize( sphereIntersection );
#endif

	// We bias the envmap to blur it under the control of the specular sharpness
	float biasFactor = max( 1.0f - specularTexColour.a, curvature );

	// Fetch the environment map colour using the reflection vector
#ifdef _3DSMAX_
	environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * biasFactor ) );
#else
	environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * biasFactor ) );
#endif

	// Calculate envmap colour (back in tangent space is fine) and add to diffuse
	accumulatedColour += environmentTexColour * vinylsEMapColour * emapFactor * fresnelCoefficient;
//	accumulatedColour = environmentTexColour;

	// Bonnetcam envmap version
#else		// ! _3DSMAX_ && BONNETCAM_ENVMAP

	// Calculate the view space reflection vector
	float3 viewSpaceNormal = mul( float4( normal, 0.0f ), view ).xyz;
	float3 viewSpaceEye = mul( float4( -eye, 0.0f ), view ).xyz;
	float3 reflectionVector = reflect( viewSpaceEye, viewSpaceNormal );

	environmentTexColour = tex2Dbias( motionBlurMap, float4( 1.0f - ( ( -reflectionVector.x * 0.5f ) + 0.5f ), 1.0f - ( ( reflectionVector.y * 0.5f ) + 0.5f ), 0.0f, 1.4f ) );

	// Calculate envmap colour (back in tangent space is fine) and add to diffuse
	accumulatedColour += environmentTexColour * vinylsEMapColour * emapFactor * fresnelCoefficient;

#endif	// defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )
	
	//
	// Perform lighting
	//
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Apply sunspot specular
#if defined( _3DSMAX_ )
	float PSL_attenuation = 1.0f;
#endif
	float3 tomtest_L = -GetLightDirection(_input);
	accumulatedColour += GetLightColour(_input) * specularTexColour * sunspotSpecularFactor * CalculateSpecularLighting( normal, tomtest_L, eye, SUNSPOT_SPECULAR_POWER ) * PSL_attenuation * ( 1.0f - curvature );

	if ( _premultiplyAlpha)
	{
		accumulatedColour.w = diffuseTexColourL1.w;
	}
	else
	{
		accumulatedColour.w = _input.colour.a;
	}
	
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Low Detail version
//
// Removes: metallic flecks, normal mapping, shadows,
//					per pixel lighting, specular lighting. Not selectable in 3DSMax.
// Keeps: Vertex lighting, vertex-shader fresnel envmapping, base layer,
//				specialisable 2nd layer, specialisable two-tone effect
//

struct VSOUTPUT_LD
{
	float4 position							: POSITION;									// View-coords position
	float4 colour								: TEXCOORD5;										// Vertex colour
	float2 baseTexCoord					: TEXCOORD0;								// Base layer texcoord
	float4 reflectionVector_plus_fresnel	: TEXCOORD1;			// Reflection vector (envmap space), with fresnel term in w
#if defined( _3DSMAX_ ) || defined( METALLIC )
	// This is only needed if we're doing metallic paintwork
	float metallicFactor				: TEXCOORD2;								// Metallic two-tone paint factor
#endif
#if defined( _3DSMAX_ ) || defined( FLIPCANDY )
	float4 normal_plus_fresnel	: TEXCOORD6;							// Normal vector (world space), with fresnel term in w
	float4 eye									: TEXCOORD7;							// Eye vector (world space)
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
	// These two coords aren't needed without layer 2
	float2 decalTexCoord				: TEXCOORD3;								// Decal map texture coord
	float2 controlTexCoord			: TEXCOORD4;								// Control map texture coord
	
	
#endif
#if defined( _3DSMAX_ ) 
	DECLARE_LIGHTING_INTERPOLATORS_VS(9)
#endif

};


VSOUTPUT_LD BodyworkLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.baseTexCoord = _input.texCoord2;

	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )


#ifdef _3DSMAX_
		// By default, Max flips the V coord of texture channels. This is because it
		// uses a BL coord system, instead of D3D's TL system. Flipping works fine for
		// wrapped textures, but for clamped textures it means the default 0...1 texture
		// space rectangle is off the object. Using 1-v instead of -v is correct in
		// both cases so we have this piece of code here, and the colorchannel declaration
		// above to do things correctly.
		_output.decalTexCoord = float2( _input.texCoord0.x, 1.0f - _input.texCoord0.y );
		_output.controlTexCoord = float2( _input.texCoord1.x, 1.0f - _input.texCoord1.y );
#else
		_output.decalTexCoord = _input.texCoord0;
		_output.controlTexCoord = _input.texCoord1;
#endif

#endif
	DEPENDENT_CODE_END( useLayer2 )

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	float3 eye = normalize( worldEyeVec );

	// if flip or candy effect is required
	DEPENDENT_CODE_START( flipcandy )
#if defined( _3DSMAX_ ) || defined( FLIPCANDY )
	// If its required
	DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
		// Calculate fresnel term
		_output.normal_plus_fresnel = float4( normal, calculated_fresnel( - normalize( worldEyeVec ), normal, fresnelFactor ) );
#endif
	DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
		// Fresnel term is 1.0
		_output.normal_plus_fresnel = float4( normal, 1.0f );
#endif
	DEPENDENT_CODE_END( useFresnel )
	_output.eye = float4(worldEyeVec,0);
#endif
	DEPENDENT_CODE_END( flipcandy )

	// If metallic paint effect is required
	DEPENDENT_CODE_START( metallic )
#if defined( _3DSMAX_ ) || defined( METALLIC )
		// Calculate metallic two-tone factor
		float ndotv = saturate( dot( normal, eye ) );
		_output.metallicFactor = pow( ndotv, metallicPower );
#endif
	DEPENDENT_CODE_END( metallic )

	// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( -eye, worldI ).xyz;
#else

#if defined( BONNETCAM_ENVMAP )
		// In bonnetcam mode, we want the world reflection vector as it'll end up being
		// transformed into view coords for calculation
		float3 envMapSpaceNormal = normal;
		float3 envMapSpaceEye = -eye;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal.xyz, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif	// defined( BONNETCAM_ENVMAP )

#endif	// defined( _3DSMAX_ )
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// If its required
		DEPENDENT_CODE_START( useFresnel )
#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
			// Calculate fresnel term
			_output.reflectionVector_plus_fresnel = float4( reflectionVector, calculated_fresnel( -eye, normal, fresnelFactor ) );
#endif
		DEPENDENT_CODE_ELSE( useFresnel )
#if defined( _3DSMAX_ ) || !defined( USE_FRESNEL )
			// Fresnel term is 1.0
			_output.reflectionVector_plus_fresnel = float4( reflectionVector, 1.0f );
#endif
		DEPENDENT_CODE_END( useFresnel )

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	{	
		DO_VS_LIGHTING_CALCULATIONS
	}	
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}



PSOUTPUT BodyworkLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read the base colour
	float4 diffuseTexColour = tex2D( diffuseMap, _input.baseTexCoord );
	float4 specularTexColour = tex2D( specularMap, _input.baseTexCoord );

	// Use the appropriate emap factor - global or vinyl one.
	// Scale down metallic paint effect on vinyls.
	// Adjust the envmapping on vinyls.
	float emapFactor = globalEMapFactor;
	float vinylsMetallicScaler = 1.0f;
	float4 vinylsEMapColour = specularTexColour;

	DEPENDENT_CODE_START( useVinyls )
#if defined( _3DSMAX_ ) || defined( USE_VINYLS )
		emapFactor = lerp( globalEMapFactor, vinylsEMapFactor, diffuseTexColour.a );
		vinylsMetallicScaler = 1.0f - diffuseTexColour.a;
		vinylsEMapColour = lerp( specularTexColour, float4( 1.0f, 1.0f, 1.0f, 1.0f ), diffuseTexColour.a );
#endif
	DEPENDENT_CODE_END( useVinyls )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// If flip or candy is required, apply it

	DEPENDENT_CODE_START( flipcandy )
#if defined( _3DSMAX_ ) || defined( FLIPCANDY )
	float3 TSnormal = normalize( _input.normal_plus_fresnel.xyz );
 	float3 eye = normalize( _input.eye.xyz );
	float ndotv = saturate( dot( TSnormal, eye ) );
	float3 _L = GetLightDirection(_input);
	float3 _H = normalize( _L + eye);
	float ndoth = saturate( dot( TSnormal, _H ) );
	float c1 = pow( ndotv, flipPower );			
	float c2 = pow( 1.0f - ndotv, flipPower );
	float c3 = pow( ndoth, candyPower );
	diffuseTexColour.rgb = c1 * diffuseTexColour.rgb + c2 * flipColour.rgb + c3 * candyColour.rgb;	
#endif
  DEPENDENT_CODE_END( flipcandy )

	// If metallic paint effect is required, apply it
	DEPENDENT_CODE_START( metallic )
#if defined( _3DSMAX_ ) || defined( METALLIC )
		diffuseTexColour.rgb += vinylsMetallicScaler * metallicColour.rgb * _input.metallicFactor;
#endif
	DEPENDENT_CODE_END( metallic )

	// If required, read and apply the decal layer over top
	float blendFactor = 0.0f;
	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		// Calculate the clamped version of texCoord0
		float2 clampedCoords = CalculateClampedCoordinates( _input.decalTexCoord, tex2D( clampControlMap, _input.controlTexCoord ) );
		float4 diffuseTexColourL2 = tex2D( diffuseMapL2, clampedCoords );

		blendFactor = diffuseTexColourL2.a;
		diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, diffuseTexColourL2.rgb, blendFactor );
#endif
	DEPENDENT_CODE_END( useLayer2 )

	// Once blended, we need to adjust vinyls-related factors - layer 2 is on top of vinyls.

	DEPENDENT_CODE_START_AND( useLayer2, useVinyls )
#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_VINYLS ) )
		emapFactor = lerp( emapFactor, globalEMapFactor, blendFactor );
		vinylsEMapColour = lerp( vinylsEMapColour, specularTexColour, blendFactor );
#endif
	DEPENDENT_CODE_END_AND( useLayer2, useVinyls )

	// Apply vertex lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;

#if defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )

	// Apply envmapping
#ifdef _3DSMAX_
	float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xzy );
#else
	float4 environmentTexColour = texCUBE( environmentMap, _input.reflectionVector_plus_fresnel.xyz );
#endif

	// Bonnetcam envmap version
#else		// defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )

	// Calculate the view space reflection vector
	float3 viewSpaceReflectionVector = mul( float4( _input.reflectionVector_plus_fresnel.xyz, 0.0f ), view ).xyz;

	float4 environmentTexColour = tex2D( motionBlurMap, float2( 1.0f - ( ( -viewSpaceReflectionVector.x * 0.5f ) + 0.5f ), 1.0f - ( ( viewSpaceReflectionVector.y * 0.5f ) + 0.5f ) ) );

#endif	// defined( _3DSMAX_ ) || !defined( BONNETCAM_ENVMAP )

	// Calculate envmap colour and add to diffuse
	accumulatedColour += environmentTexColour * vinylsEMapColour * emapFactor * _input.reflectionVector_plus_fresnel.w;

	accumulatedColour.a = diffuseTexColour.a;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Bodywork
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Bodywork";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Bodywork_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   usefullprecision = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx BodyworkVertexShader();
		PixelShader = compile sce_fp_rsx BodyworkFragmentShader( false );
#else		
		VertexShader = compile vs_3_0 BodyworkVertexShader();
		PixelShader = compile ps_3_0 BodyworkFragmentShader( false );
#endif		
	}
}



technique Bodywork_Translucent
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Bodywork_Translucent";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Bodywork_LowDetail";
	int    lowDetailDeferredID	= 2;
	bool   usefullprecision = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx BodyworkVertexShader();
		PixelShader = compile sce_fp_rsx BodyworkFragmentShader( true );
#else		
		VertexShader = compile vs_3_0 BodyworkVertexShader();
		PixelShader = compile ps_3_0 BodyworkFragmentShader( true );
#endif		
	}
}



technique Bodywork_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Bodywork_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BodyworkLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx BodyworkLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 BodyworkLowDetailVertexShader();
		PixelShader = compile ps_3_0 BodyworkLowDetailFragmentShader();
#endif
	}
}
